package it.softecspa.s4fs.vfat;

import it.softecspa.kahuna.util.calendar.EnterpriseCalendar;
import it.softecspa.s4fs.S4Config;
import it.softecspa.s4fs.S4Engine;

import org.apache.log4j.Logger;




/**
 * Thread di cancellazioni
 * 
 * @author m.veroni
 *
 */
public class FilesystemCleaner implements Runnable {

	private Logger log = Logger.getLogger(getClass());
	
	private S4Config config;
	private S4Engine engine;
	
	private Thread iThread;
	
	
	
	
	public FilesystemCleaner(S4Engine engine) {
		this.engine = engine;
		this.config = engine.getConfig();
		log.info("Cleaner thread ready to start");
	}
	
	
	public void run() {
		iThread = Thread.currentThread();
		// Non faccio partire subito il thread, aspetto 60 secondi
		log.info("Cleaner startup sleep (60 seconds)");
		try {
			Thread.sleep(60*1000);
		} catch (InterruptedException e) {
			log.error("Thread interrupted with exception", e);
			stop();
			return;
		}	
		
		
			
		while (true) {
			// --------------------- Inizio ----------------------------
			
			execute();			
			
			// --------------------- Fine ------------------------------
			// Metto in pausa il thread
			long millis = config.getCleanerDelay()*1000; 
			if (log.isDebugEnabled()) {
				EnterpriseCalendar sveglia = EnterpriseCalendar.now();
				sveglia.add(EnterpriseCalendar.SECOND, config.getCleanerDelay());
				log.debug("Cleaner thread sleep until "+sveglia.formatISO8601withTimezone());
			}
			try {
				Thread.sleep(millis);
			} catch (InterruptedException e) {
				log.error("Thread interrupted with exception", e);
				stop();
				return;
			}		
			
			if (log.isDebugEnabled()) log.debug("Cleaner thread is awake");
		}
	}

	
	/**
	 * Processo effettivo ci cancellazione
	 */
	private synchronized void execute() {
		if (log.isDebugEnabled()) log.debug("Cleaning in progress....");		
		VFATEngine vfat = engine.getVFAT();
		
		// FASE 1 - Pulizia di tutti i file su filesystem locale
		if (log.isDebugEnabled()) log.debug("Cleaning file on "+Storage.LOCAL+" storage");
		vfat.cleaningExpiredFilesLOCAL();		
		
		// FASE 2 - Pulizia di tutti i file che risiedono su filesystem Amazon S3
		if (engine.useS3Service()) {
			if (log.isDebugEnabled()) log.debug("Cleaning file on "+Storage.S3+" storage");
			vfat.cleaningExpiredFilesS3(vfat.getBucket().getUid());
		}
		
		// FASE 3 - Cancellazione di tutti i record marcati come DRAFT pi vecchi di 24 ore
		if (log.isDebugEnabled()) log.debug("Expire DRAFT record");
		int ret = vfat.expireDraftRecord(1);
		if (log.isDebugEnabled()) log.debug("Marked "+ret+" records");
				
		
		if (log.isDebugEnabled()) log.debug("Cleaning finished!");
	}


	public void stop() {
		if (iThread==null) return;		
		try {
			iThread.interrupt();
		} catch (Exception e) {
			log.error(e.getClass().getSimpleName()+" interrupting cleaner thread", e);
		}
		log.info("Cleaner thread stopped");
	}
	
	
	
}
