package it.softecspa.s4fs.vfat;

import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.TimeZone;

import org.apache.log4j.Logger;


/**
 * Classe utilizzata come semaforo di sincronizzazione per cluster
 * Utilizza come elemento a comune il filesystem
 * 
 * @author m.veroni 
 */
public class LockerS4 {

	private Logger log = Logger.getLogger(getClass());
	private String reference;
	
	private VFATEngine vfat;	
	private FileAllocation vfile;
	private FileS4 lock;
	
	public static final String PATTERN_UTC_ZULU = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'";
	
	private final long DEFAULT_LONG_CHECK = 1000*60*60;   // 1 ora
	private final long DEFAULT_SHORT_CHECK = 1000*60*20;  // 20 minuti;
	
	private long longTimeout;
	private long shortTimeout;
	
	
	LockerS4(VFATEngine vfat, FileAllocation vfile, String reference) {
		this.vfat = vfat;
		this.vfile = vfile;
		this.reference = reference;
	}
	
	
	/**
	 * Rilascia il file di lock
	 * @return
	 */
	public boolean release() {
		if (lock==null) return true;
		
		synchronized (LockerS4.class) {			
			return vfat.releaseLock(this);			
		}
	}

	
	
	
	/**
	 * Acquisisce il lock sulle coordinate 
	 * @return
	 */
	public synchronized boolean acquire() {
		return acquire(DEFAULT_SHORT_CHECK, DEFAULT_LONG_CHECK);
	}
	
	
	/**
	 * Acquisisce il lock passando i timeout di controllo
	 * @param shortTimeout
	 * @param longTimeout
	 * @return
	 */
	public synchronized boolean acquire(long shortTimeout, long longTimeout) {
		synchronized (LockerS4.class) {			
			this.shortTimeout = shortTimeout;
			this.longTimeout = longTimeout;
			vfat.lockAcquire(this);
			return (lock!=null);
		}		
	}

	
	private boolean compareTimestamp(Timestamp value1, Timestamp value2, long check, String note) {
		/*
		 * La data letta dal file "deve" essere nel passato rispetto alla timestamp attuale
		 */				
		long differenza = value1.getTime()-value2.getTime();
		differenza = Math.abs(differenza);
		
		boolean ret = (differenza>=check);
		if (ret) log.warn("[" + getLockName() + "] Waiting time too long (timeout "+(check/1000)+" seconds) "+note);
		return ret;
	}
	
	boolean checkShortTimestamp(Timestamp timestamp) {
		return compareTimestamp(vfile.getLock(), timestamp, shortTimeout, "'short timeout'");
	}
	
	boolean checkLongTimestamp(Timestamp timestamp) {
		return compareTimestamp(vfile.getLock(), timestamp, longTimeout, "'long timeout'");
	}

	String getLockName() {
		return vfile.getContext()+vfile.getNesting()+vfile.getName();
	}
	
	
	
	

	/**
	 * Restituisce TRUE se il lock  acquisito
	 * @return
	 */
	public boolean isLocked() {
		return (lock!=null);
	}

	/**
	 * Coordinate della allocazione virtuale utilizzata per il lock
	 * @return
	 */
	public FileAllocation getCoordinate() {
		return vfile;
	}

	
	/**
	 * Istanza di {@link FileS4} utilizzata per il lock
	 * Attenzione: il file restituito non  un file utilizzabile 
	 * @return
	 */
	public FileS4 getLock() {
		return lock;
	}

	void setLock(FileS4 lock) {
		this.lock = lock;
	}


	/**
	 * Referenza del lock
	 * @return
	 */
	public String getReference() {
		return reference;
	}
	
	
	
	static String formatTimestamp(Timestamp value) {
		if (value==null) return "";
		
		SimpleDateFormat dateParser = new SimpleDateFormat(PATTERN_UTC_ZULU);
		dateParser.setTimeZone(TimeZone.getTimeZone("UTC"));	
		return dateParser.format(value.getTime());
	}
	
	
	
	
}
