package it.softecspa.s4fs.vfat;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Calendar;

import org.apache.log4j.Logger;

import com.amazonaws.AmazonClientException;
import com.amazonaws.AmazonServiceException;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.AccessControlList;
import com.amazonaws.services.s3.model.DeleteObjectRequest;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.PutObjectResult;
import com.amazonaws.services.s3.model.S3Object;

import it.softecspa.kahuna.util.calendar.EnterpriseCalendar;
import it.softecspa.s4fs.S4Engine;
import it.softecspa.s4fs.S4Exception;
import it.softecspa.s4fs.S4Status;

public class S3FileAction {

	private Logger log = Logger.getLogger(getClass());
	
	private S4Engine engine;
	private FileS4 file;
	
	
	public S3FileAction(FileS4 file) throws S4Exception {
		engine = S4Engine.getInstance();
		if (!engine.isConfigured()) throw new S4Exception(S4Status.CONFIGURATION_PROBLEM);
		
		this.file = file;
	}


	public synchronized void uploadFile() throws S4Exception {	
		if (!engine.useS3Service()) throw new S4Exception(S4Status.IMPOSSIBLE_DOWNLOAD);
		
		try {
			AmazonS3 s3 = engine.getS3Service();
			String bucketName = engine.getS4bucket().getName();
			String key = file.getS3Key();
			
			PutObjectResult res = putS3ObjectWithStatistics(s3, new PutObjectRequest(bucketName, key, file.getRealFile()));
			// Sono sicuro che il file  stato depositato su S3
			if (log.isDebugEnabled()) log.debug("File uploaded with success on Amazon S3: ETag '"+res.getETag()+"'");
	        
			try {
				/* Aggiorno le policy in funzione di quelle impostate
				 * - se sono stare cambiate 
				 * - se nell'upload precedente si era verificato un errore
				 */
				if (!file.getS3Grant().equals(file.getCompareS3Grant()) ||
					 file.getS3Status()!=0) {
					if (log.isDebugEnabled()) log.debug("Grant new permission to file: '"+file.getS3Grant()+"'");
			        
					// Recupero i permessi dati al file caricato
		            AccessControlList acl = s3.getObjectAcl(bucketName, key);
					
					// Ciclo su tutti i GroupGrantee e Sas3Permission
					for (GroupGrantee grantee : GroupGrantee.values()) {	
						// ...azzero i precedenti grant del gruppo
						acl.revokeAllPermissions(grantee.getAmazon());
						// ...aggiungo la permission
						for (Permission permission : Permission.values()) {
							if (file.isS3Grant(grantee, permission)) {
								if (log.isDebugEnabled()) log.debug("Add permission "+permission.toString()+" on group "+grantee.toString());
								acl.grantPermission(grantee.getAmazon(), permission.getAmazon());
							}
						}
					}
					s3.setObjectAcl(bucketName, key, acl);            
				}
				
			// In questo caso l'errore  segnalato con un semplice warning e sono aggiornati i valori su db	
			} catch (AmazonServiceException e) {
				log.warn("Problem updating object ACL, reset to default",e);
				file.resetS3Grant();
		    } catch (AmazonClientException e) {
		    	log.warn("Problem updating object ACL, reset to default",e);
		    	file.resetS3Grant();   		    	
		    } catch (Exception e) {
		    	log.warn("Problem updating object ACL, reset to default",e);
		    	file.resetS3Grant();
			}			
		    
		    // Aggiorno il record con le nuove informazioni
			file.setBucketUid(engine.getS4bucket().getUid());
			file.setS3ETag(res.getETag());
			file.setS3VersionId(res.getVersionId());
	        
		} catch (AmazonServiceException e) {
			throw new S4Exception(e);
	    } catch (AmazonClientException e) {
	    	throw new S4Exception(e);            	
	    }	 
		
		
	}
	
	
	public synchronized void downloadFile() throws S4Exception {		
		if (!engine.useS3Service()) throw new S4Exception(S4Status.IMPOSSIBLE_DOWNLOAD);
		
		AmazonS3 s3 = engine.getS3Service();
		String bucketName = engine.getS4bucket().getName();
		String key = file.getS3Key();
		
		try {
			S3Object object = getS3ObjectWithStatistics(s3, new GetObjectRequest(bucketName, key));
			// Ho recuperato il file da S3
			if (log.isDebugEnabled()) log.debug("File downloaded with success on Amazon S3: ETag '"+object.getObjectMetadata().getETag()+"'");
        
			
			FileOutputStream fos = null;
			BufferedReader reader = null;
			boolean complete_with_success = false;
			try {
				fos = new FileOutputStream(file.getRealFile());
				reader = new BufferedReader(new InputStreamReader(object.getObjectContent()));
				
				int c = reader.read();
			    while (c != -1){
			      c = reader.read();
			      fos.write(c);
			    }	
			    complete_with_success = true;
			    
			} catch (FileNotFoundException e) {
			   throw new S4Exception(S4Status.REAL_FILE_NOT_FOUND,e);			        
			} catch (IOException e){
				throw new S4Exception(S4Status.REAL_FILE_WRITE_ERROR,e);
			} catch (Exception e) {
				throw new S4Exception(S4Status.UNHANDLED_EXCEPTION,e);
			} finally {
				
				try {
					if (reader!=null) reader.close();
				} catch (IOException e) {
					log.warn("Problem closing BufferedReader");
				}
				try {
					if (fos!=null) fos.close();
				} catch (IOException e) {
					log.warn("Problem closing FileOutputStream");
				}
				
				if (complete_with_success) {
					file.removeFile();
				}
			}
		
		} catch (AmazonServiceException e) {
			throw new S4Exception(e);
	    } catch (AmazonClientException e) {
	    	throw new S4Exception(e);            	
	    }	
			
	}
	
	public synchronized void deleteFile() throws S4Exception {		
		if (!engine.useS3Service()) throw new S4Exception(S4Status.IMPOSSIBLE_DOWNLOAD);
		
		AmazonS3 s3 = engine.getS3Service();
		String bucketName = engine.getS4bucket().getName();
		String key = file.getS3Key();
		
		try {
			delteS3ObjectWithStatistics(s3, new DeleteObjectRequest(bucketName, key));
			// Ho cancellato il file da S3
			log.info("File '"+file.getS3Key()+"' deleted with success on Amazon S3");
		} catch (AmazonServiceException e) {
			throw new S4Exception(e);
	    } catch (AmazonClientException e) {
	    	throw new S4Exception(e);            	
	    }	
			
	}
	
	
	private PutObjectResult putS3ObjectWithStatistics(AmazonS3 s3, PutObjectRequest request) {
		Calendar _start= null;
		if (log.isDebugEnabled()) {
			_start = Calendar.getInstance();
		}
		PutObjectResult res = s3.putObject(request);
		if (log.isDebugEnabled()) {
			Calendar _stop = Calendar.getInstance();
			log.debug("Upload time: "+EnterpriseCalendar.newInstance(_stop).difference(new EnterpriseCalendar(_start), Calendar.SECOND)+" seconds");
		}
		return res;		
	}
	
	private void delteS3ObjectWithStatistics(AmazonS3 s3, DeleteObjectRequest request) {
		Calendar _start= null;
		if (log.isDebugEnabled()) {
			_start = Calendar.getInstance();
		}
		s3.deleteObject(request);
		if (log.isDebugEnabled()) {
			Calendar _stop = Calendar.getInstance();
			log.debug("Delete time: "+EnterpriseCalendar.newInstance(_stop).difference(new EnterpriseCalendar(_start), Calendar.SECOND)+" seconds");
		}
	}
	
	private S3Object getS3ObjectWithStatistics(AmazonS3 s3, GetObjectRequest request) {
		Calendar _start= null;
		if (log.isDebugEnabled()) {
			_start = Calendar.getInstance();
		}
		S3Object res = s3.getObject(request);
		if (log.isDebugEnabled()) {
			Calendar _stop = Calendar.getInstance();
			log.debug("Download time: "+EnterpriseCalendar.newInstance(_stop).difference(new EnterpriseCalendar(_start), Calendar.SECOND)+" seconds");
		}
		return res;		
	}


	public S4Engine getEngine() {
		return engine;
	}


	public FileS4 getFile() {
		return file;
	}
	
	
}
