package it.softecspa.s4fs.vfat;

import it.softecspa.database.dbconnect.ConnectionManager;
import it.softecspa.kahuna.util.calendar.EnterpriseCalendar;
import it.softecspa.s4fs.S4Bucket;
import it.softecspa.s4fs.S4Exception;
import it.softecspa.s4fs.S4Status;

import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Calendar;
import java.util.TimeZone;

import org.apache.log4j.Logger;


/**
 * Virtual File Allocation Table
 * @author m.veroni
 */
public class VFATEngine  {

	private static VFATEngine instance;
	
	public static VFATEngine newInstance(ConnectionManager cmWritable, Owner owner, S4Bucket bucket) {
		synchronized (VFATEngine.class) {
			if (instance == null) new VFATEngine(cmWritable, owner, bucket);			
		}
		return instance;
	}
	
	private Logger log = Logger.getLogger(getClass());
	
	
	// Identificativo univo del nodo che utilizzer la Virtual FAT
	private Owner owner;
	private S4Bucket bucket;
	private VirtualAllocationTable at;
	private GarbageAllocationTable atd;

	
	private VFATEngine(ConnectionManager cm, Owner owner, S4Bucket bucket) {
		super();
		synchronized (VFATEngine.class) {
			if (instance == null) {
				instance = this;
				
				// Inizializzazione variabili
				this.at = new VirtualAllocationTable(cm);
				this.atd = new GarbageAllocationTable(cm);
				this.owner = owner;
				this.bucket = bucket;	// Attenzione questo pu essere nullo
				
				log.info(VFATEngine.class.getSimpleName()+" instance is starting!");	
			}
		}
		
		
	}
	
	
	
	
	/**
	 * Restitutuisce TRUE se il file ricercato esiste
	 * In caso di risposta positiva l'oggetto in ingresso  valorizzato contutti i valori recuprati dall VFAT
	 * @param file
	 * @return
	 * @throws VFATException 
	 */
	public boolean existsFile(String context, String nesting, String name) throws VFATException {
		if (log.isDebugEnabled()) log.debug("Check if file exists on VFAT");
		
		FileAllocation allocation = new FileAllocation();
		allocation.setContext(context);
		allocation.setNesting(nesting);
		allocation.setName(name);
		
		FileS4 file = at.load(allocation);	
		if (file!=null) return true;
		return false;
	}
	
	
	/**
	 * Recupera dalla VFAT le info relative al file cercato
	 * @param file
	 * @return
	 * @throws S4Exception 
	 * @throws VFATException 
	 */
	public FileS4 loadFile(String filesystemRoot, String context, String nesting, String name) throws VFATException, S4Exception {
		return loadFile(filesystemRoot, context, nesting, name, WritePolicy.READONLY);
	}
	
	public FileS4 loadFile(String filesystemRoot, String context, String nesting, String name, WritePolicy writable) throws VFATException, S4Exception {
		if (log.isDebugEnabled()) log.debug("Load file from VFAT");	
		
		FileAllocation allocation = new FileAllocation();
		allocation.setContext(context);
		allocation.setNesting(nesting);
		allocation.setName(name);
		
		FileS4 file = at.load(allocation);	
		if (file==null) {
			// il file non esiste
			return null;
		}
		
		if (file.isDraft()) {
			// Il file  gi presente in modalit DRAFT			
			throw new VFATException("File in draft mode; open with create: "+VFATException.coordinate(file));
		}
		if (file.isLocked()) {
			// Il file  gi presente in modalit DRAFT			
			throw new VFATException("File in draft mode; open with create: "+VFATException.coordinate(file));
		}
		
		
		// Creo l'istanza reale del File
		file.createRealFile(filesystemRoot);
				
		// Sincronizzo il file con il filesystem S3 se configurato
		syncronizeFile(filesystemRoot, file);
		
		// Il file creato  buono, posso utilizzarlo
		file.setWritePolicy(writable);		
		file.setChanged(false);	   // Il file sar modifcato su S3 soltanto in modo esplicito
		file.setOpen(true);
		return file;	
	}
	
	/**
	 * Estrae le informazioni relative al file
	 * @param filesystemRoot
	 * @param uid
	 * @return
	 * @throws VFATException
	 */
	public FileS4 fileInfo(String filesystemRoot, String context, String nesting, String name) throws VFATException {
		if (log.isDebugEnabled()) log.debug("Load file info from VFAT by name");
		
		FileAllocation allocation = new FileAllocation();
		allocation.setContext(context);
		allocation.setNesting(nesting);
		allocation.setName(name);
		
		FileS4 file = at.load(allocation);	
		if (file==null) {
			// il file non esiste
			return null;
		}
		
		// Creo l'istanza reale del File
		file.createRealFile(filesystemRoot);
		
		return file;	
	}
	
	
	public FileS4 fileInfo(String filesystemRoot, Integer uid, boolean garbage) throws VFATException {
		if (log.isDebugEnabled()) log.debug("Load file info from VFAT by uid");
		
		FileAllocation allocation = new FileAllocation();
		allocation.setUid(uid);
		
		FileS4 file = null;
		if (garbage) {
			file = atd.load(allocation);
		} else {
			file = at.load(allocation);
		}
		if (file==null) {
			// il file non esiste
			return null;
		}
		
		// Creo l'istanza reale del File
		file.createRealFile(filesystemRoot);
		
		return file;	
	}
	

	public Owner getOwner() {
		return owner;
	}



	/**
	 * Esegue un controllo di consistenza della VFAT
	 * @throws SQLException
	 */
	public void check() throws VFATException {
		if (log.isDebugEnabled()) log.debug("Check VFAT");
		// TODO check della VFAT, da implementare
		
	}


	public VFATStatistics contextStatistic(String context) throws VFATException {
		if (log.isDebugEnabled()) log.debug("Recovery context VFAT statistics");
		return at.statistics(context);		
	}
	
	
	
	
	public synchronized FileS4 createFile(String filesystemRoot, String context, String parent, String name, FilesystemPolicy fsPolicy, WritePolicy wp) throws VFATException {
		if (log.isDebugEnabled()) log.debug("Create new file");
		return manageFile(filesystemRoot, context, parent, name, fsPolicy, wp, true);
	}
		
		
	/**
	 * Esegue le operazioni di creazione file sulla sulla VFAT 
	 * @param filesystemRoot
	 * @param context
	 * @param nesting
	 * @param name
	 * @param fsPolicy
	 * @param writePolicy
	 * @param overwriteLocal
	 * @param copyFrom
	 * @return
	 * @throws VFATException
	 */
	private synchronized FileS4 manageFile(String filesystemRoot, String context, String nesting, String name, FilesystemPolicy fsPolicy, WritePolicy writePolicy, boolean overwriteLocal) throws VFATException {
		 
		checkFileName(name);
		
		
		FileAllocation allocation = new FileAllocation();
		allocation.setOwnerUid(owner.getUid());
		allocation.setContext(context);
		allocation.setNesting(nesting);
		allocation.setName(name);
		// Politiche di salvataggio file
		allocation.setFSPolicy(fsPolicy);		
		
		FileS4 file = at.create(allocation);		
		if (!file.isDraft()) {
			// Il file  gi presente e creato in altro contesto		
			if (!writePolicy.equals(WritePolicy.UPDATABLE)) {
				// ...non posso sovrascrivare il file
				throw new VFATException("VFAT duplicate creation request: "+VFATException.coordinate(file));
			}
			
		} else {
			// Il file esite come DRAFT,  stato appena creato
			if (!file.getOwnerUid().equals(owner.getUid())) {
				// La DRAFT  stata creata da un altro server, quindi non la posso utilizzare
				throw new VFATException("Different draft owner ("+file.getOwnerUid()+" vs "+owner.getUid()+"): "+VFATException.coordinate(file));
			}
		}
		
		// Creo l'istanza reale del File
		file.createRealFile(filesystemRoot);
				
		
		
		if (file.getRealFile().exists()) {	
			if (overwriteLocal && writePolicy.equals(WritePolicy.UPDATABLE)) {
				// Preventivamente cancello il vecchio file
				VFATEngine.deleteLocalFylesystem(file);
			} else {
				// Il file esite gi, non posso proseguire
				throw new VFATException("File duplicated on local filesystem: '"+file.getRealFile().getPath()+"'");
			}
		}
		
		
		// Il file creato  buono, posso utilizzarlo
		file.setWritePolicy(WritePolicy.UPDATABLE);		
		file.setChanged(true);	   
		file.setOpen(true);
		
		return file;		
	}
	
	
	// Controlla se il nome del file  valido
	public void checkFileName(String name) throws VFATException {
		
		if (name.indexOf("/")>0) 	throw new VFATException("Character / (slash) is not valid in file name");
		if (name.indexOf("\\")>0) 	throw new VFATException("Character \\ (backslash) is not valid in file name");
		if (name.indexOf("*")>0) 	throw new VFATException("Character * (asterix) is not valid in file name");
		if (name.indexOf("!")>0) 	throw new VFATException("Character ! is not valid in file name");
		if (name.indexOf("?")>0) 	throw new VFATException("Character ? is not valid in file name");
		if (name.indexOf(" ")>0) 	throw new VFATException("Character blank is not valid in file name");
		if (name.indexOf("'")>0) 	throw new VFATException("Character ' is not valid in file name");
		if (name.indexOf("\"")>0) 	throw new VFATException("Character \" is not valid in file name");
	}




	protected static void deleteLocalFylesystem(FileS4 file) throws VFATException {
		try {
			if (!file.getRealFile().delete()) {
				throw new VFATException("Existent file do not deleted: '"+file.getRealFile().getPath()+"'");				
			}
		} catch (SecurityException e) {
			throw new VFATException("Security exception in delete overwritable file: '"+file.getRealFile().getPath()+"'", e);
		} catch (Exception e) {
			throw new VFATException("Unhandled exception in delete overwritable file: '"+file.getRealFile().getPath()+"'", e);
		}
		
	}




	/**
	 * Consolida il file sulla VFAT e contestualmente lo marca come chiuso
	 * @param file
	 * @throws VFATException
	 */
	private void store(FileS4 file) throws VFATException {
		// ...tutti i metodi sono sincronizzati sul file
		if (file==null) throw new NullPointerException();
		synchronized (file) {
		
			if (log.isDebugEnabled()) log.debug("Store data on VFAT record");
			at.save(file);
			
		}
	}
	
	
	/**
	 * Aggiorna il record dell'allocazione con le info dell'operazione su Amazon S3
	 * @param file
	 * @param status
	 * @param message
	 * @throws VFATException
	 */
	private void status(FileS4 file, int status, String message) throws VFATException {
		if (log.isDebugEnabled()) log.debug("Update status of VFAT record");
		file.setS3Status(status);
		file.setS3StatusMessage(message);
		at.status(file);	
	}



	/**
	 * Sincronizza il file locale recuperandolo da Amazon S3
	 * @param filesystemRoot
	 * @param file
	 * @throws VFATException
	 * @throws S4Exception
	 */
	public synchronized void syncronizeFile(String filesystemRoot, FileS4 file) throws VFATException, S4Exception {
		if (log.isDebugEnabled()) log.debug("Synchronize file from delocalizated filesystem");
		
		// Controllo le policy di storage
		boolean download = false;
		if (file.getFSPolicy().useLocal() || file.getFSPolicy().useCache()) {
			if (!file.getRealFile().exists()) {
				if (!file.getFSPolicy().useS3()) {
					throw new VFATException("File not found on local filesystem: "+VFATException.coordinate(file));
				}
				download = true;
			}	
		}		
				
		if (!download) {
			// TODO Il file  fi presente su filesystem locale, controllare la sua validit
			
			
		}
		if (download) {
			if (!file.isS3Shared()) {
				throw new VFATException("File not found on local filesystem (not shared): "+VFATException.coordinate(file));
			}		
			
			log.info("Download file from Amazon S3 volume to "+file.getRealFile().getPath());
			S3FileAction s3Action = new S3FileAction(file);		
			s3Action.downloadFile();
		}
		
		
		// Aggiorno l'eventuale data di scadenza automatica
		if (file.getFSExpirationCache()>0) {
			at.updateExpireCACHE(file, owner);
		}
		
	}




	public synchronized void closeFile(FileS4 file) throws S4Exception {
		if (log.isDebugEnabled()) log.debug("Close file on VFAT");
		
		// ...tutti i metodi sono sincronizzati sul file
		if (file==null) throw new NullPointerException();
		synchronized (file) {
		
			// Controlli formali su realFile
			if (file.getRealFile()==null) throw new S4Exception(S4Status.REAL_FILE_NULL);
			if (!file.getRealFile().exists()) throw new S4Exception(S4Status.REAL_FILE_NOT_EXIST,": '"+file.getRealFile().getPath()+"'" );
			
			// Aggiorno i dati da salvare sulla VFAT
			if (file.isDraft()) {
				// Devo valorizzare anche le date di creazione e modifica
				file.setCreation(EnterpriseCalendar.now().getTimestamp());
				file.setModifierUid(file.getOwnerUid());
				file.setModify(file.getCreation());		
				file.setDraft(false); 	// Il file non  pi draft
			} else {
				// Aggiorno le date di modifica
				file.setModifierUid(file.getOwnerUid());
				file.setModify(EnterpriseCalendar.now().getTimestamp());
			}
			// Aggiorno la dimansione del file
			file.setSize(file.getRealFile().length());		
			
			
			// Consolido i dati sulla VFAT perche ormai il file locale  stato creato
			try {			
				store(file);			
			} catch (VFATException e) {
				throw new S4Exception(S4Status.VFAT_EXCEPTION, e);
			}
					
			
			// Gestione policy di cancellazione automatica (fase 1)
			try {
				if (log.isDebugEnabled()) log.debug("Check and set storage expiration policy ("+Storage.LOCAL+" and "+Storage.CACHE+")");
				at.updateExpire(Storage.LOCAL, owner, file);
				at.updateExpire(Storage.CACHE, owner, file);
			} catch (VFATException e) {
				log.error(S4Status.FILE_EXPIRATION_SETUP.toString(), e);
				//...e vado avanti
			} catch (Exception e) {
				throw new S4Exception(S4Status.UNHANDLED_EXCEPTION, e);
			}
			
			
			
			// ...proseguo con la verifica dell info su S3
			if (file.getFSPolicy().useS3()) {
				if (log.isDebugEnabled()) log.debug("File has to be stored on Amazon S3 volume");
				// il file deve essere condiviso su S3			
				
				/* Controllo se:
				 * - il file  stato cambiato
				 * - in precedenza c' stato un errore nell'upload su Amazon S3
				 */
				if (file.isChanged() || 
					file.getS3Status()!=0) { 
					try {
						S3FileAction s3Action = new S3FileAction(file);
						s3Action.uploadFile();			
					} catch (S4Exception e) {
						try {
							status(file, e.getStatusCode(), e.getMessage());
						} catch (VFATException e1) {
							throw new S4Exception(S4Status.VFAT_EXCEPTION, e);
						}				
						throw e;
					}
				
					
					// Gestione policy di cancellazione automatica (fase 2)
					try {
						if (log.isDebugEnabled()) log.debug("Check and set storage expiration policy ("+Storage.S3+")");
						at.updateExpire(Storage.S3, owner, file);
					} catch (VFATException e) {
						log.error(S4Status.FILE_EXPIRATION_SETUP.toString(), e);
						//...e vado avanti
					} catch (Exception e) {
						throw new S4Exception(S4Status.UNHANDLED_EXCEPTION, e);
					}
					
					
					// Consolido i dati sulla VFAT perche ormai il file locale  stato creato
					try {			
						store(file);
					} catch (VFATException e) {				
						throw new S4Exception(S4Status.VFAT_EXCEPTION, e);
					}	
					
				} else {
					if (log.isDebugEnabled()) log.debug("No change found in file, no upload on Amazon S3 volume");
				}
			}
			
			
			if (!file.getFSPolicy().useLocal()) {
				if (log.isDebugEnabled()) log.debug("File has to be delete from local filesystem");
				// Il file deve essere cancellato dall'area locale			
				file.removeFile();
			}
					
			
			// Pulizia definitiva del file se di tipo BUFFER
			if (file.getFSPolicy()==FilesystemPolicy.BUFFER) {
				// Sposto il record nel garbage
				try {
					if (log.isDebugEnabled()) log.debug("Delete file from VFAT (Garbage)");
					at.delete(file, true);
				} catch (VFATException e) {
					throw new S4Exception(S4Status.VFAT_EXCEPTION,e);
				}
			}
			
			
			// Da adesso in poi considero il file CHIUSO, e quindi non posso modificare POLICY e TIMEOUT
			file.setWritePolicy(WritePolicy.READONLY);
			file.setChanged(false);
			file.setOpen(false);
		}
	}



	/**
	 * Cancella fisicamente i file "scaduti" su filesystem S3
	 * Nessuna eccezione  propagata
	 * @param local
	 */
	public boolean cleaningExpiredFilesS3(int bucketUid) {
		if (log.isDebugEnabled()) log.debug("Cleaning expired file from "+Storage.S3+" storage");
		try {
			at.cleaningExpiredS3(bucketUid, EnterpriseCalendar.now().getTimestamp());
			return true;
		} catch (VFATException e) {
			log.error("Unthrowable VFATException cleaning expired files from "+Storage.S3+" storage");
		}
		return false;
	}
	
	public boolean cleaningExpiredFilesLOCAL() {
		if (log.isDebugEnabled()) log.debug("Cleaning expired file from "+Storage.LOCAL+" storage");
		try {
			at.cleaningExpiredLOCAL(owner, EnterpriseCalendar.now().getTimestamp());
			return true;
		} catch (VFATException e) {
			log.error("Unthrowable VFATException cleaning expired files from "+Storage.LOCAL+" storage");
		}
		return false;
	}




	public S4Bucket getBucket() {
		return bucket;
	}

	public void setBucket(S4Bucket bucket) {
		this.bucket = bucket;
	}



	/**
	 * Cacella il file dalla VFAT e schedula una cancellazione da tutti i sistemi
	 * @param executor 
	 * @param file
	 * @throws VFATException 
	 */
	public void deleteFile(Owner executor, FileS4 file) throws S4Exception {
		// ...tutti i metodi sono sincronizzati sul file
		if (file==null) throw new NullPointerException();
		synchronized (file) {
			
			// Sposto il record nel garbage
			try {
				if (log.isDebugEnabled()) log.debug("Delete file from VFAT (Garbage)");
				at.delete(file, true);
			} catch (VFATException e) {
				throw new S4Exception(S4Status.VFAT_EXCEPTION,e);
			}
			
			// Se il file si trova su Amazon S3 lo cancello subito da li
			if (file.isS3Shared()) {
				S3FileAction action = new S3FileAction(file);
				action.deleteFile();
			}
			
			// Rimuovo il file dal filesystem locale
			file.removeFile();
			
			
			// Schedulo la cancellazione immediata sugli altri nodi (tutti quelli in anagrafica)
			if (file.getFSPolicy().isDistributable()) {
				try {
					at.expireNowDistributedFiles(executor, file);
				} catch (VFATException e) {
					throw new S4Exception(S4Status.VFAT_EXCEPTION,e);
				}
			}
		}
		
	}



	/**
	 * Recupera un handle di lock alle coordinate richieste
	 * Il lock si appoggia sulla VFAT
	 * @param context
	 * @param nesting
	 * @param name
	 * @param reference
	 * @return
	 */
	public LockerS4 getLockHandle(String context, String nesting, String name, String reference) {
		FileAllocation vfile = new FileAllocation();
		vfile.setOwnerUid(owner.getUid());
		vfile.setContext(context);
		vfile.setNesting(nesting);
		vfile.setName(name);
		
		vfile.setLockerUid(owner.getUid());
		/* Info accessorie
		 * ...per semplicit  stato reciclato il campo s3_status_message
		 */		
		vfile.setS3StatusMessage("Lock reference: "+reference);
		
		return new LockerS4(this, vfile, reference);
	}



	/**
	 * Acquisisce il lock
	 * @param locker
	 */
	public void lockAcquire(LockerS4 locker) {
		if (log.isDebugEnabled()) log.debug("("+locker.getReference()+") Acquire lock with key '"+locker.getLockName()+"' now!");
		try {
			Timestamp timestamp = new Timestamp(Calendar.getInstance(TimeZone.getTimeZone("UTC")).getTimeInMillis());
			at.acquireLock(locker, timestamp);			
		} catch (VFATException e) {
			log.error("("+locker.getReference()+") Unthrowable VFATException acquiring with key '"+locker.getLockName()+"'");
		}		
	}


	/**
	 * Rilascia il lock precedentemete impostato
	 * @param locker
	 * @return
	 */
	public boolean releaseLock(LockerS4 locker) {
		if (log.isDebugEnabled()) log.debug("("+locker.getReference()+") Release lock with key '"+locker.getLockName()+"' now!");
		try {
			at.lockRelease(locker.getLock());
		} catch (VFATException e) {
			log.error("("+locker.getReference()+") Unthrowable VFATException relearing with key '"+locker.getLockName()+"'",e);
		}
		return false;
	}



	/** 
	 * Fa scadere i file inserit come DRAFT ormai da alcuni giorni 
	 * @param i
	 */
	int expireDraftRecord(int dayDelay) {
		if (log.isDebugEnabled()) log.debug("Expire DRAFT record insert "+dayDelay+" days ago");
		try {
			Calendar after = Calendar.getInstance();
			after.add(Calendar.DAY_OF_MONTH, -dayDelay);
			return at.expireDraft(owner, new Timestamp(after.getTimeInMillis()));
		} catch (VFATException e) {
			log.error("Unthrowable VFATException expiring draft record",e);
		}
		return 0;	
	}




	








	




	
	
	
	
	
	
	
	
}
