package it.softecspa.s4fs.vfat;

import it.softecspa.database.dbconnect.ConnectionManager;
import it.softecspa.database.dbconnect.DatabaseStatementUnplugged;
import it.softecspa.kahuna.lang.XString;
import it.softecspa.kahuna.util.calendar.EnterpriseCalendar;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import com.mysql.jdbc.MysqlErrorNumbers;

/**
 * Gestore della Virtual File Allocation Table
 * 
 * @author m.veroni
 * 
 */
public class VirtualAllocationTable extends Autocreate {

	private final String NAME = "s4_010_vfat";

	private ConnectionManager cm;
	
	
	public VirtualAllocationTable(ConnectionManager cm) {
		this.cm = cm;
	}
	
	
	/**
	 * Costruice la WHERE condition in base ai filtri passati
	 * @return
	 */
	@SuppressWarnings("unused")
	private StringBuilder buildWhere(FileAllocation filter) {
		List<String> conditions = new ArrayList<String>();

		// Chiave primaria
		if (filter.uid != null)
			conditions.add("uid = " + filter.uid.toString());

		// Chiave logica
		if (filter.context != null)
			conditions.add("context = " + Format.toString(filter.getContext()));
		if (filter.nesting != null)
			conditions.add("nesting = " + Format.toString(filter.getNesting()));
		if (filter.name != null)
			conditions.add("name = " + Format.toString(filter.getName()));

		StringBuilder where = new StringBuilder(50);
		for (String contidion : conditions) {
			if (where.length() == 0) {
				where.append(" WHERE ").append(contidion);
			} else {
				where.append(" AND ").append(contidion);
			}
		}
		return where;
	}

	private final String COLUMNS = "uid, owner, context, nesting, name, creation, modifier, modify, size, storage_local, storage_s3, storage_cache, timeout_local, timeout_s3, timeout_cache, s3_bucket, s3_grant, s3_etag, s3_versionid, draft, locker, locking, s3_status, s3_status_message";

	
	/**
	 * Crea un nuovo record nella tabella della allocazioni in modo esclusivo e sincrtonizzato
	 * @param vfile
	 * @return
	 * @throws VFATException
	 */
	public synchronized FileS4 create(FileAllocation vfile) throws VFATException {
		try {
			DatabaseStatementUnplugged dbsu = DatabaseStatementUnplugged.getInstance(cm); // con transazione
			try {
				// ---------------------------- CORE del metodo ------------------------------
				
				StringBuilder sql = new StringBuilder(100);
				sql.append("SELECT ").append(COLUMNS);
				sql.append(" FROM ").append(NAME);
				sql.append(" WHERE context = ").append(Format.toString(vfile.getContext()));
				sql.append(" AND nesting = ").append(Format.toString(vfile.getNesting()));
				sql.append(" AND name = ").append(Format.toString(vfile.getName()));
				sql.append(" FOR UPDATE");
				try {
					ResultSet res = dbsu.getResultSet(sql.toString());
					if (res.next()) {
						// Restituisco il file estratto
						return new FileS4(res);
					}
				} catch (SQLException e1) {
					checkAndCreate(dbsu, e1);
				}

				/*
				 * Il file non esiste, posso crearlo e restituirlo il file  marcato come DRAFT per indicare che non  ancora consolidato
				 */
				FileS4 file = new FileS4(vfile, true);
				executeSave(dbsu, file);
				dbsu.commit(); // Chiudo la transazione e libero il lock
				return file;

				// ---------------------------- CORE del metodo ------------------------------				
			} catch (SQLException e) {
				throw new VFATException(e);
			} finally {
				dbsu.free();
			}
		} catch (SQLException e) {
			throw new VFATException(e);
		} catch (Exception e) {
			throw new VFATException("Unhandled exception", e);
		}
	}

	
	public synchronized void acquireLock(LockerS4 locker, Timestamp timestamp) throws VFATException {
		try {
			DatabaseStatementUnplugged dbsu = DatabaseStatementUnplugged.getInstance(cm); // con transazione
			try {
				// ---------------------------- CORE del metodo ------------------------------
				dbsu.hideQueryLog(); 	// Nessul SQL stampato nel log
				
				FileAllocation vfile = locker.getCoordinate();
				vfile.setLock(new Timestamp(Calendar.getInstance().getTimeInMillis()));
				FileS4 prelock = null;
				
				
				StringBuilder sql = new StringBuilder(100);
				sql.append("SELECT ").append(COLUMNS);
				sql.append(" FROM ").append(NAME);
				sql.append(" WHERE context = ").append(Format.toString(vfile.getContext()));
				sql.append(" AND nesting = ").append(Format.toString(vfile.getNesting()));
				sql.append(" AND name = ").append(Format.toString(vfile.getName()));
				sql.append(" FOR UPDATE");
				try {
					ResultSet res = dbsu.getResultSet(sql.toString());
					if (res.next()) {
						// Restituisco il file estratto
						prelock = new FileS4(res);
					}
				} catch (SQLException e1) {
					checkAndCreate(dbsu, e1);
				}

				if (prelock!=null) {					
					if (prelock.getLockerUid()==null || prelock.getLockerUid()==null) {
						// il FileS4 non  un file di lock
						throw new VFATException("("+locker.getReference()+") This is not a lock record: '"+locker.getLockName()+"'");
					}
					
					/*
					 * Il file di lock  gi presente, controllo OWNER e LOCKING
					 * per l'eventuale sblocco automatico
					 */
					if (vfile.getOwnerUid().equals(prelock.getOwnerUid())) {
						// Il lock  dell OWNER, controllo il primo timeout
						if (!locker.checkShortTimestamp(prelock.getLock())) {
							log.info("("+locker.getReference()+") Another process use this lock ["+prelock.getLockerUid()+","+LockerS4.formatTimestamp(prelock.getLock())+"]");
							return;
						}	
					} else {
						// Il lock  stato impostato da un'altro sistema, controllo il secondo timeout 
						if (!locker.checkLongTimestamp(prelock.getLock())) {
							log.info("("+locker.getReference()+") Another process use this lock ["+prelock.getLockerUid()+","+LockerS4.formatTimestamp(prelock.getLock())+"]");
							return;	
						}
					}
					
					// ... passato troppo tempo, rilascio il lock 
					int uid = prelock.getUid();							
					prelock = new FileS4(vfile, false);
					prelock.setUid(uid);
					
				} else {
					/*
					 * Il file non esiste, posso crearlo e restituirlo
					 * Il LOCK non  DRAFT
					 */
					prelock = new FileS4(vfile, false);
				}				
				
				executeSave(dbsu, prelock);
				dbsu.commit(); // Chiudo la transazione e libero il lock sulla tabella
				locker.setLock(prelock);
				
				// ---------------------------- CORE del metodo ------------------------------
			} catch (SQLException e) {
				throw new VFATException(e);

			} finally {
				dbsu.free();
			}
		} catch (SQLException e) {
			throw new VFATException(e);
		} catch (Exception e) {
			throw new VFATException("Unhandled exception", e);
		}
	}
	
	
	
	
	
	
	public synchronized void save(FileS4 file) throws VFATException {
		try {
			DatabaseStatementUnplugged dbsu = DatabaseStatementUnplugged.getInstanceAUTOCOMMIT(cm); // senza transazione
			try {
				// ---------------------------- CORE del metodo ------------------------------
				
				executeSave(dbsu, file);
				
				// ---------------------------- CORE del metodo ------------------------------
			} catch (SQLException e) {
				throw new VFATException(e);
			} finally {
				dbsu.free();
			}
		} catch (SQLException e) {
			throw new VFATException(e);
		} catch (Exception e) {
			throw new VFATException("Unhandled exception", e);
		}
	}

	static class Format {

		public static String toString(String value) {
			return (value == null ? "null" : "'" + value + "'");
		}

		public static String toString(boolean value) {
			return (value ? "1" : "0");
		}

		public static String toString(Date value) {
			return (value == null ? "null" : "'" + EnterpriseCalendar.newInstance(value).format("yyyyMMdd")) + "'";
		}

		public static String toString(Timestamp value) {
			return (value == null ? "null" : "'" + EnterpriseCalendar.newInstance(value).format("yyyyMMddHHmmss") + "'");
		}

		public static String getTimestamp() {
			return "now()";
		}
	}

	/**
	 * Salva il record sulla VFAT
	 * 
	 * @param dbsu
	 * @param file
	 * @throws SQLException
	 * @throws VFATException
	 */
	private void executeSave(DatabaseStatementUnplugged dbsu, FileS4 file) throws SQLException, VFATException {
		if (file.getUid()==null) {
			
			// Inserisco il nuovo record
			StringBuilder sql = new StringBuilder(200);			
			sql.append("INSERT INTO ").append(NAME).append(" (").append(COLUMNS).append(")");
			sql.append(" VALUES (null");
			sql.append(", ").append(file.getOwnerUid());
			sql.append(", ").append(Format.toString(file.getContext()));
			sql.append(", ").append(Format.toString(file.getNesting()));
			sql.append(", ").append(Format.toString(file.getName()));
			sql.append(", ").append(Format.toString(file.getCreation()));
			sql.append(", ").append(file.getModifierUid());
			sql.append(", ").append(Format.toString(file.getModify()));
			sql.append(", ").append(file.getSize());
			sql.append(", ").append(Format.toString(file.getFSPolicy().useLocal()));
			sql.append(", ").append(Format.toString(file.getFSPolicy().useS3()));
			sql.append(", ").append(Format.toString(file.getFSPolicy().useCache()));
			sql.append(", ").append(file.getFSExpirationLocal());
			sql.append(", ").append(file.getFSExpirationS3());
			sql.append(", ").append(file.getFSExpirationCache());
			sql.append(", ").append(file.getBucketUid());
			sql.append(", ").append(Format.toString(file.getS3Grant()));
			sql.append(", ").append(Format.toString(file.getS3ETag()));
			sql.append(", ").append(Format.toString(file.getS3VersionId()));
			sql.append(", ").append(Format.toString(file.isDraft()));
			sql.append(", ").append(file.getLockerUid());
			sql.append(", ").append(Format.toString(file.getLock()));
			sql.append(", ").append(0);
			sql.append(", ").append("null");
			sql.append(")");
			try {
				dbsu.execute(sql.toString());			
			} catch (SQLException e1) {
				checkAndCreate(dbsu, e1);
				dbsu.execute(sql.toString());	
			}
			
			// Eseguo una select per recuperare l'UID
			sql = new StringBuilder(100);
			sql.append("SELECT uid");
			sql.append(" FROM ").append(NAME);
			sql.append(" WHERE context = ").append(Format.toString(file.getContext()));
			sql.append(" AND nesting = ").append(Format.toString(file.getNesting()));
			sql.append(" AND name = ").append(Format.toString(file.getName()));
			ResultSet res = dbsu.getResultSet(sql.toString());
			if (!res.next()) {
				throw new VFATException("Problem in recover VFAT uid: "+VFATException.coordinate(file));
			}
			file.setUid(res.getInt("uid"));
			
			
		} else {
			// Aggiorno il record
			StringBuilder sql = new StringBuilder(200);			
			sql.append("UPDATE ").append(NAME).append(" SET");
			sql.append("  context = ").append(Format.toString(file.getContext()));
			sql.append(", nesting = ").append(Format.toString(file.getNesting()));
			sql.append(", name = ").append(Format.toString(file.getName()));
			sql.append(", creation = ").append(Format.toString(file.getCreation()));
			sql.append(", modifier = ").append(file.getModifierUid());
			sql.append(", modify = ").append(Format.toString(file.getModify()));
			sql.append(", size = ").append(file.getSize());
			sql.append(", storage_local = ").append(Format.toString(file.getFSPolicy().useLocal()));
			sql.append(", storage_s3 = ").append(Format.toString(file.getFSPolicy().useS3()));
			sql.append(", storage_cache = ").append(Format.toString(file.getFSPolicy().useCache()));
			sql.append(", timeout_local = ").append(file.getFSExpirationLocal());
			sql.append(", timeout_s3 = ").append(file.getFSExpirationS3());
			sql.append(", timeout_cache = ").append(file.getFSExpirationCache());
			sql.append(", s3_bucket = ").append(file.getBucketUid());
			sql.append(", s3_grant = ").append(Format.toString(file.getS3Grant()));
			sql.append(", s3_etag = ").append(Format.toString(file.getS3ETag()));
			sql.append(", s3_versionid = ").append(Format.toString(file.getS3VersionId()));
			sql.append(", draft = ").append(Format.toString(file.isDraft()));
			sql.append(", locker = ").append(file.getLockerUid());
			sql.append(", locking = ").append(Format.toString(file.getLock()));
			sql.append(", s3_status = 0");
			sql.append(", s3_status_message = null");
			sql.append(" WHERE uid = ").append(file.getUid());
			if (dbsu.execute(sql.toString())!=1) throw new VFATException("No record updated with uid = "+file.getUid());
		}
		
	}
	
	


	/**
	 * Aggiorna lo stato della allocazione
	 * @param file
	 * @throws VFATException
	 */
	public synchronized void status(FileS4 file) throws VFATException {
		try {
			DatabaseStatementUnplugged dbsu = DatabaseStatementUnplugged.getInstanceAUTOCOMMIT(cm); // senza transazione
			try {
				// ---------------------------- CORE del metodo ------------------------------
				
				// Aggiorno il record
				StringBuilder sql = new StringBuilder(200);			
				sql.append("UPDATE ").append(NAME).append(" SET");
				sql.append("  s3_status = ").append(file.getS3Status());
				sql.append(", s3_status_message = ").append(Format.toString(XString.cut(file.getS3StatusMessage(),512)));
				sql.append(" WHERE uid = " + file.getUid());
				if (dbsu.execute(sql.toString())!=1) throw new VFATException("No record updated with uid = "+file.getUid());
				
				// ---------------------------- CORE del metodo ------------------------------
			} catch (SQLException e) {
				throw new VFATException(e);
			} finally {
				dbsu.free();
			}
		} catch (SQLException e) {
			throw new VFATException(e);
		} catch (Exception e) {
			throw new VFATException("Unhandled exception", e);
		}
		
	}

	/* TODO loadFileWithLock da implementare
	public FileS4 loadWithLock(FileAllocation allocation) {		
		return null;
	}
	*/
	
	
	public FileS4 load(FileAllocation vfile) throws VFATException {
		
		try {
			DatabaseStatementUnplugged dbsu = DatabaseStatementUnplugged.getInstanceAUTOCOMMIT(cm); // senza transazione
			try {
				// ---------------------------- CORE del metodo ------------------------------
				
				StringBuilder sql = new StringBuilder(100);
				sql.append("SELECT ").append(COLUMNS);
				sql.append(" FROM ").append(NAME);
				if (vfile.getUid()!=null) {
					sql.append(" WHERE uid = ").append(vfile.getUid());
				} else {
					sql.append(" WHERE context = ").append(Format.toString(vfile.getContext()));
					sql.append(" AND nesting = ").append(Format.toString(vfile.getNesting()));
					sql.append(" AND name = ").append(Format.toString(vfile.getName()));
				}
				try {
					ResultSet res = dbsu.getResultSet(sql.toString());
					if (res.next()) {
						// Restituisco il file estratto
						return new FileS4(res);
					}
				} catch (SQLException e1) {
					checkAndCreate(dbsu, e1);
				}

				/*
				 * Il file non esiste
				 */
				return null;

				// ---------------------------- CORE del metodo ------------------------------
			} catch (SQLException e) {
				throw new VFATException(e);
			} finally {
				dbsu.free();
			}
		} catch (SQLException e) {
			throw new VFATException(e);
		} catch (Exception e) {
			throw new VFATException("Unhandled exception", e);
		}
	}
	
	
	public synchronized boolean delete(FileAllocation vfile, boolean garbage) throws VFATException {		
		try {
			DatabaseStatementUnplugged dbsu = DatabaseStatementUnplugged.getInstanceAUTOCOMMIT(cm); // senza transazione
			try {
				// ---------------------------- CORE del metodo ------------------------------
				
				return delete(dbsu, vfile, garbage);

				// ---------------------------- CORE del metodo ------------------------------
			} catch (SQLException e) {
				throw new VFATException(e);
			} finally {
				dbsu.free();
			}
		} catch (SQLException e) {
			throw new VFATException(e);
		} catch (Exception e) {
			throw new VFATException("Unhandled exception", e);
		}
	}

	
	public synchronized boolean lockRelease(FileAllocation vfile) throws VFATException {		
		try {
			DatabaseStatementUnplugged dbsu = DatabaseStatementUnplugged.getInstanceAUTOCOMMIT(cm); // senza transazione
			try {
				// ---------------------------- CORE del metodo ------------------------------
				dbsu.hideQueryLog(); 	// Nessu SQL stampato nel log
				
				return delete(dbsu, vfile, false);

				// ---------------------------- CORE del metodo ------------------------------
			} catch (SQLException e) {
				throw new VFATException(e);
			} finally {
				dbsu.free();
			}
		} catch (SQLException e) {
			throw new VFATException(e);
		} catch (Exception e) {
			throw new VFATException("Unhandled exception", e);
		}
	}
	
	private synchronized boolean delete (DatabaseStatementUnplugged dbsu, FileAllocation vfile, boolean garbage) throws SQLException {
		if (garbage) {
			// Sposto il record nella tabella di garbage
			StringBuilder sql = new StringBuilder(150);
			sql.append("INSERT INTO ").append(GarbageAllocationTable.NAME).append("(").append(COLUMNS).append(")");
			sql.append("SELECT ").append(COLUMNS);
			sql.append(" FROM ").append(NAME);
			sql.append(" WHERE context = ").append(Format.toString(vfile.getContext()));
			sql.append(" AND nesting = ").append(Format.toString(vfile.getNesting()));
			sql.append(" AND name = ").append(Format.toString(vfile.getName()));
			try {
				dbsu.execute(sql.toString());
			} catch (SQLException e1) {
				/* Controllo se ho un errore di chiave duplicata
				 * In questo caso passo oltre, non  un problema
				 */
				if (e1.getErrorCode()!=MysqlErrorNumbers.ER_DUP_KEY) {
					checkAndCreate(dbsu, e1);
				}
			}
		}
		
		// Cancello il record
		StringBuilder sql = new StringBuilder(150);
		sql.append("DELETE ");
		sql.append(" FROM ").append(NAME);
		sql.append(" WHERE context = ").append(Format.toString(vfile.getContext()));
		sql.append(" AND nesting = ").append(Format.toString(vfile.getNesting()));
		sql.append(" AND name = ").append(Format.toString(vfile.getName()));
		try {
			int ret = dbsu.execute(sql.toString());
			if (ret==0) {
				log.warn("Strange! Allocation not present on VFAT: "+VFATException.coordinate(vfile));
			}
		} catch (SQLException e1) {
			checkAndCreate(dbsu, e1);
		}
		
		return true;
	}

	@Override
	String createSQL() {
		StringBuilder sql = new StringBuilder(200);
		sql.append("CREATE TABLE ").append(NAME).append(" (");
		sql.append(" uid int(10) UNSIGNED NOT NULL AUTO_INCREMENT,");
		sql.append(" owner int(10) UNSIGNED NOT NULL,");
		sql.append(" context varchar(100) NOT NULL,");
		sql.append(" nesting varchar(200) NOT NULL,");
		sql.append(" name varchar(100) NOT NULL,");
		sql.append(" creation datetime NULL,");
		sql.append(" modifier int(10) UNSIGNED NULL,");
		sql.append(" modify datetime NULL,");
		sql.append(" size int(10) UNSIGNED NULL,");
		sql.append(" storage_local tinyint(1) UNSIGNED NOT NULL DEFAULT 0,");
		sql.append(" storage_s3 tinyint(1) UNSIGNED NOT NULL DEFAULT 0,");
		sql.append(" storage_cache  tinyint(1) UNSIGNED NOT NULL DEFAULT 0,");
		sql.append(" timeout_local int(10) UNSIGNED NOT NULL DEFAULT 0,");
		sql.append(" timeout_s3 int(10) UNSIGNED NOT NULL DEFAULT 0,");
		sql.append(" timeout_cache int(10) UNSIGNED NOT NULL DEFAULT 0,");
		sql.append(" s3_bucket int(10) UNSIGNED NULL,");
		sql.append(" s3_grant varchar(20) NULL,");
		sql.append(" s3_etag varchar(100) NULL,");
		sql.append(" s3_versionid varchar(100) NULL,");
		sql.append(" draft tinyint(1) UNSIGNED NOT NULL DEFAULT 0,");
		sql.append(" locker int(10) UNSIGNED NULL,");
		sql.append(" locking datetime NULL,");
		sql.append(" s3_status int(6) NOT NULL DEFAULT 0,");
		sql.append(" s3_status_message varchar(512) NULL,");
		sql.append(" timerecord timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,");
		sql.append(" primary key (uid), ");
		sql.append(" unique index iu_vfat_logic_key (context, nesting, name)");
		sql.append(" ) ENGINE = InnoDB");
		sql.append(" COMMENT = 'Virtual File Allocation Table'");
		return sql.toString();
	}


	@Override
	public String getTableName() {
		return NAME;
	}

	
	/**
	 * Aggiorna la data di scadenza per i file sullo storage indicato
	 * @param storage
	 * @param owner
	 * @param file
	 * @throws VFATException
	 */
	public void updateExpire(Storage storage, Owner owner, FileS4 file) throws VFATException {
		int i=storage.index();
		
		// ...controllo se....
		if (file.getFSExpiration()[i]!=file.getCompareFSExpiration()[i]) {
			// ...sono cambiate!
			if (i==Storage.LOCAL.index()) {
				if (file.getOwnerUid().equals(owner)) {
					updateExpireLOCAL(file);
				}
			} else if (i==Storage.S3.index()) {
				updateExpireS3(file);
				
			} else if (i==Storage.CACHE.index()) {
				if (!file.getOwnerUid().equals(owner)) {
					updateExpireCACHE(file, owner);
				}					
			}
			
		}
	
	}

	protected void updateExpireS3(FileS4 file) throws VFATException {
		if (file.getFSExpirationS3()<0) return;
		
		// Inserisco un record per Amazon S3
		ExpireAllocation record = new ExpireAllocationS3(file.getUid(), file.getBucketUid(), file.getFSExpirationS3());
		record.update(cm);
		if (log.isDebugEnabled()) log.debug("Storage "+Storage.S3+" expiration date is "+EnterpriseCalendar.newInstance(record.getExpire()).formatISO8601withTimezone());		
	}
	
	protected void updateExpireLOCAL(FileS4 file) throws VFATException {
		if (file.getFSExpirationLocal()<0) return;
		
		// Sono il proprietario, inserico un record per il server locale
		ExpireAllocation record = new ExpireAllocationFS(file.getUid(), file.getOwnerUid(), file.getFSExpirationLocal());
		record.update(cm);
		if (log.isDebugEnabled()) log.debug("Storage "+Storage.LOCAL+" expiration date is "+EnterpriseCalendar.newInstance(record.getExpire()).formatISO8601withTimezone());
	}
	
	protected void updateExpireCACHE(FileS4 file, Owner owner) throws VFATException {
		if (file.getFSExpirationCache()<0) return;
		
		// Non sono il proprietario, inserico un record per la cache
		ExpireAllocation record = new ExpireAllocationFS(file.getUid(), owner.getUid(), file.getFSExpirationCache());
		record.update(cm);
		if (log.isDebugEnabled()) log.debug("Storage "+Storage.CACHE+" expiration date is "+EnterpriseCalendar.newInstance(record.getExpire()).formatISO8601withTimezone());
	}


	/**
	 * Toglie i file dallo storage locale 
	 * @param owner
	 * @param expired
	 * @throws VFATException
	 */
	protected void cleaningExpiredLOCAL(Owner owner, Timestamp expired) throws VFATException {
		ExpireAllocationFS allocation = new ExpireAllocationFS(owner.getUid(), expired);
		allocation.deleteExpired(cm);
	}
	
	
	/**
	 * Toglie i file dallo storage Amazon S3
	 * @param bucketUid
	 * @param expired
	 * @throws VFATException
	 */
	protected void cleaningExpiredS3(int bucketUid, Timestamp expired) throws VFATException {
		ExpireAllocationS3 allocation = new ExpireAllocationS3(bucketUid, expired);
		allocation.deleteExpired(cm);
	}


	/**
	 * Fa "scadere" i file distribuiti sui filesystem dei nodi
	 * @param exclusion
	 * @param file
	 * @throws VFATException
	 */
	public void expireNowDistributedFiles(Owner exclusion, FileS4 file) throws VFATException  {
		ExpireAllocationFS.expireNowWithExclude(file, true, cm, exclusion);
	}


	/**
	 * Estrae dei dati statistici dalla VFAT
	 * @return
	 * @throws VFATException 
	 */
	public VFATStatistics statistics(String context) throws VFATException {
		VFATStatistics stats = new VFATStatistics(context);
		try {
			DatabaseStatementUnplugged dbsu = DatabaseStatementUnplugged.getInstanceAUTOCOMMIT(cm); // senza transazione
			try {
				// ---------------------------- CORE del metodo ------------------------------
				
				ResultSet res = null;
				
				// RECORD COUNT
				stats.setRecord(countContextAllocations(dbsu, context));
				
				
				// DRAFT FILE
				StringBuilder sql = new StringBuilder(50);
				sql.append("SELECT count(1)");
				sql.append(" FROM ").append(NAME);
				sql.append(" WHERE context = ").append(Format.toString(context));
				sql.append(" AND draft = 1");
				res = dbsu.getResultSet(sql.toString());
				if (res.next()) {
					// Restituisco il file estratto
					stats.setDraft(res.getInt(1));
				}
			
				
				// LOCKED FILE
				sql = new StringBuilder(50);
				sql.append("SELECT count(1)");
				sql.append(" FROM ").append(NAME);
				sql.append(" WHERE context = ").append(Format.toString(context));
				sql.append(" AND locking is not null");				
				res = dbsu.getResultSet(sql.toString());
				if (res.next()) {
					// Restituisco il file estratto
					stats.setLocked(res.getInt(1));
				}
				
				// S3 SHARED
				sql = new StringBuilder(50);
				sql.append("SELECT count(1)");
				sql.append(" FROM ").append(NAME);
				sql.append(" WHERE context = ").append(Format.toString(context));
				sql.append(" AND s3_etag is not null");				
				res = dbsu.getResultSet(sql.toString());
				if (res.next()) {
					// Restituisco il file estratto
					stats.setS3shared(res.getInt(1));
				}
				
				
				// GARBAGE
				GarbageAllocationTable garbage = new GarbageAllocationTable(cm);
				stats.setGarbage(garbage.countContextAllocations(dbsu, context));
				
				
				// ---------------------------- CORE del metodo ------------------------------
			} catch (SQLException e) {
				throw new VFATException(e);
			} finally {
				dbsu.free();
			}
		} catch (SQLException e) {
			throw new VFATException(e);
		} catch (Exception e) {
			throw new VFATException("Unhandled exception", e);
		}
		
		
		return stats;
	}
	
	
	
	public int countContextAllocations(DatabaseStatementUnplugged dbsu, String context) throws SQLException {
		ResultSet res = null;
		StringBuilder sql = new StringBuilder(50);
		sql.append("SELECT count(1)");
		sql.append(" FROM ").append(getTableName());
		sql.append(" WHERE context = ").append(Format.toString(context));		
		try {
			res = dbsu.getResultSet(sql.toString());
		} catch (SQLException e1) {
			checkAndCreate(dbsu, e1);
			res = dbsu.getResultSet(sql.toString());
		}
		if (res.next()) {
			// Restituisco il file estratto
			return res.getInt(1);
		}
		return 0;
	}

	
	/**
	 * Fa scadere i file allocati come DRAFT ormai pi vecchi di un certo periodo
	 * @param owner delle allocazioni
	 * @param dayDelay
	 * @return
	 * @throws VFATException 
	 */
	public int expireDraft(Owner owner, Timestamp after) throws VFATException {
		try {
			DatabaseStatementUnplugged dbsu = DatabaseStatementUnplugged.getInstanceAUTOCOMMIT(cm); // con transazione
			try {
				// ---------------------------- CORE del metodo ------------------------------
				int count=0;
				
				StringBuilder sql = new StringBuilder(100);
				sql.append("SELECT ").append(COLUMNS);
				sql.append(" FROM ").append(NAME);
				sql.append(" WHERE owner = ").append(owner.getUid());				// in mio possesso
				sql.append(" AND draft = 1");										// bozze
				sql.append(" AND locking is null");									// non bloccati
				sql.append(" AND timerecord <= ").append(Format.toString(after)); 	// pi vecchi di 20 ore
				try {
					ResultSet res = dbsu.getResultSet(sql.toString());
					while (res.next()) {
						// Cancello il file, spostandolo nel garbage
						FileS4 file = new FileS4(res);
						delete(dbsu, file, true);
						
						// Inserisco il file estratto nell'elenco degli "expired" in modo da cancellari eventuali files appesi su filesystem
						ExpireAllocationFS expire = new ExpireAllocationFS(res.getInt("uid"), owner.getUid(), after, true);
						expire.update(dbsu);						
						
						count++;
					}
				} catch (SQLException e1) {
					checkAndCreate(dbsu, e1);
				}
				return count;
				
				// ---------------------------- CORE del metodo ------------------------------				
			} catch (SQLException e) {
				throw new VFATException(e);
			} finally {
				dbsu.free();
			}
		} catch (SQLException e) {
			throw new VFATException(e);
		} catch (Exception e) {
			throw new VFATException("Unhandled exception", e);
		}
	}


	
	
	
}
